/*:
 * @target MZ
 * @plugindesc v1.2.3 オプションに【セーブ／ロード／タイトルへ】を追加 & 多言語連動／タイトルのオプションでは非表示／セーブ安定化
 * @author HS
 *
 * @help
 * ◇機能
 *  - ゲーム中のオプションに「セーブ」「ロード」「タイトルへ」を追加（タイトル画面のオプションでは非表示）。
 *  - 決定入力の持ち越しをクリアして安全に遷移。
 *  - セーブ/ロードの成功・失敗をコンソール表示。
 *  - セーブ直前に保存内容を「安全なクローン」に変換し、PIXI/Spine等の実体参照を除去。
 *    ※ クローン時に null プロトタイプを Object.prototype へフォールバック（JsonEx対策）。
 *
 * ◇注意
 *  - 実体(Spineプレイヤー/PIXI DisplayObject/関数/DOM等)は $gameTemp 等の非保存領域に置いてください。
 *  - 既読/オート等「入力を横取り」する系は Scene_File 中では無効化してください。
 *
 * 推奨並び順：HS_LangSimple →（他の表記ローカライズ）→ 本プラグイン → 入力フック系（既読/オート/スキップ等）
 *
 * @param showSave
 * @text セーブを表示（ゲーム中のみ）
 * @type boolean
 * @default true
 *
 * @param showLoad
 * @text ロードを表示（ゲーム中のみ）
 * @type boolean
 * @default true
 *
 * @param showTitle
 * @text タイトルに戻るを表示（ゲーム中のみ）
 * @type boolean
 * @default true
 *
 * @param saveFb0
 * @text セーブ(0:日本語)
 * @default セーブ
 * @param saveFb1
 * @text セーブ(1:English)
 * @default Save Game
 * @param saveFb2
 * @text セーブ(2:简体中文)
 * @default 存档
 * @param saveFb3
 * @text セーブ(3:한국어)
 * @default 세이브
 *
 * @param loadFb0
 * @text ロード(0:日本語)
 * @default ロード
 * @param loadFb1
 * @text ロード(1:English)
 * @default Load Game
 * @param loadFb2
 * @text ロード(2:简体中文)
 * @default 读档
 * @param loadFb3
 * @text ロード(3:한국어)
 * @default 로드
 *
 * @param titleFb0
 * @text タイトルへ(0:日本語)
 * @default タイトルへ
 * @param titleFb1
 * @text タイトルへ(1:English)
 * @default Title Screen
 * @param titleFb2
 * @text タイトルへ(2:简体中文)
 * @default 标题画面
 * @param titleFb3
 * @text タイトルへ(3:한국어)
 * @default 타이틀 화면
 */

(() => {
  'use strict';

  // ---- parameters
  const pluginName = (document.currentScript && document.currentScript.src)
    ? decodeURIComponent(document.currentScript.src.split('/').pop().replace(/\.js$/,''))
    : 'HS_UtilityOptions_Localize';

  const P = PluginManager.parameters(pluginName);
  const SHOW_SAVE   = P.showSave === 'true';
  const SHOW_LOAD   = P.showLoad === 'true';
  const SHOW_TITLE  = P.showTitle === 'true';
  const saveFb = [P.saveFb0||"", P.saveFb1||"", P.saveFb2||"", P.saveFb3||""];
  const loadFb = [P.loadFb0||"", P.loadFb1||"", P.loadFb2||"", P.loadFb3||""];
  const titleFb= [P.titleFb0||"", P.titleFb1||"", P.titleFb2||"", P.titleFb3||""];

  const SYM_SAVE  = "hsSave";
  const SYM_LOAD  = "hsLoad";
  const SYM_TITLE = "hsToTitle";

  // ---- language (HS_LangSimple 連動)
  function langIndex(){
    try { return Number(ConfigManager && ConfigManager.hs_langIndex || 0) || 0; }
    catch(_){ return 0; }
  }
  const labelSave  = () => (saveFb[langIndex()]  || saveFb[0]  || "Save");
  const labelLoad  = () => (loadFb[langIndex()]  || loadFb[0]  || "Load");
  const labelTitle = () => (titleFb[langIndex()] || titleFb[0] || "Title");

  // ---- 状況判定
  function inTitle(){ return SceneManager._scene instanceof Scene_Title; }
  function inBattle(){ return $gameParty && $gameParty.inBattle && $gameParty.inBattle(); }

  function canSaveNow(){
    if (inTitle()) return false;
    if (inBattle()) return false;
    try { return !!($gameSystem && $gameSystem.isSaveEnabled && $gameSystem.isSaveEnabled()); }
    catch(_){ return false; }
  }
  function canLoadNow(){ return true; }
  function canToTitleNow(){
    if (inTitle()) return false;
    if (inBattle()) return false;
    return true;
  }

  // === ゲーム中かどうか（タイトルでは false） ===
  function hsInGame(){
    try { return $gameMap && typeof $gameMap.mapId === 'function' && $gameMap.mapId() > 0; }
    catch(_) { return false; }
  }

  // ---- オプション項目の追加（ゲーム中のみ表示）
  const _addGeneralOptions = Window_Options.prototype.addGeneralOptions;
  Window_Options.prototype.addGeneralOptions = function(){
    _addGeneralOptions.apply(this, arguments);
    if (!hsInGame()) return; // タイトルのオプションでは非表示

    if (SHOW_SAVE)  this.addCommand(labelSave(),  SYM_SAVE,  canSaveNow());
    if (SHOW_LOAD)  this.addCommand(labelLoad(),  SYM_LOAD,  canLoadNow());
    if (SHOW_TITLE) this.addCommand(labelTitle(), SYM_TITLE, canToTitleNow());
  };

  // ---- ステータス欄(右側)は空表示
  const _statusText = Window_Options.prototype.statusText;
  Window_Options.prototype.statusText = function(index){
    const sym = this.commandSymbol(index);
    if (sym === SYM_SAVE || sym === SYM_LOAD || sym === SYM_TITLE) return "";
    return _statusText.call(this, index);
  };

  // ---- 入力持ち越しを防いで安全にシーンを開く
  function openSceneSafely(sceneClass){
    const scene = SceneManager._scene;
    if (scene && scene._optionsWindow && scene._optionsWindow.deactivate){
      scene._optionsWindow.deactivate();
    }
    if (window.Input && Input.clear) Input.clear();
    if (window.TouchInput && TouchInput.clear) TouchInput.clear();
    setTimeout(() => SceneManager.push(sceneClass), 0); // 次フレームへ
  }

  // ---- 決定処理の差し替え（セーブ/ロード/タイトルへ）
  const _processOk = Window_Options.prototype.processOk;
  Window_Options.prototype.processOk = function(){
    const idx = this.index();
    const sym = this.commandSymbol(idx);
    if (sym === SYM_SAVE || sym === SYM_LOAD || sym === SYM_TITLE){
      if (!this.isCommandEnabled(idx)){ SoundManager.playBuzzer(); return; }
      this.playOkSound();
      if (sym === SYM_SAVE)  { openSceneSafely(Scene_Save);   return; }
      if (sym === SYM_LOAD)  { openSceneSafely(Scene_Load);   return; }
      if (sym === SYM_TITLE) { openSceneSafely(Scene_GameEnd); return; }
    }
    _processOk.call(this);
  };

  // ---- 言語切替に追従してラベル更新
  function relabel(win){
    if (!win || !win._list) return;
    for (const it of win._list){
      if (!it) continue;
      if (it.symbol === SYM_SAVE)  it.name = labelSave();
      if (it.symbol === SYM_LOAD)  it.name = labelLoad();
      if (it.symbol === SYM_TITLE) it.name = labelTitle();
    }
  }
  const _optRefresh = Window_Options.prototype.refresh;
  Window_Options.prototype.refresh = function(){
    try{ relabel(this); }catch(_){}
    _optRefresh.call(this);
  };

  // ---- セーブ/ロード画面のリストを確実にアクティブに
  const _sceneFileStart = Scene_File.prototype.start;
  Scene_File.prototype.start = function(){
    _sceneFileStart.call(this);
    if (this._listWindow && !this._listWindow.active) this._listWindow.activate();
  };

  // ---- セーブ/ロード成否のログ（原因特定用）
  const _executeSave = Scene_File.prototype.executeSave;
  Scene_File.prototype.executeSave = function(savefileId){
    try { $gameSystem.onBeforeSave(); } catch(e){ console.error('onBeforeSave error:', e); }
    const p = DataManager.saveGame(savefileId);
    if (p && p.then){
      p.then(() => { console.info('[HS] save success:', savefileId); this.onSaveSuccess(); })
       .catch(err => { console.error('[HS] save FAILED:', err); SoundManager.playBuzzer(); this.onSaveFailure(); });
    } else {
      const ok = !!p;
      console[ok?'info':'error']('[HS] save ' + (ok?'success':'FAILED') + ':', savefileId);
      ok ? this.onSaveSuccess() : this.onSaveFailure();
    }
  };
  const _executeLoad = Scene_File.prototype.executeLoad;
  Scene_File.prototype.executeLoad = function(savefileId){
    const p = DataManager.loadGame(savefileId);
    if (p && p.then){
      p.then(() => { console.info('[HS] load success:', savefileId); this.onLoadSuccess(); })
       .catch(err => { console.error('[HS] load FAILED:', err); SoundManager.playBuzzer(); this.onLoadFailure(); });
    } else {
      const ok = !!p;
      console[ok?'info':'error']('[HS] load ' + (ok?'success':'FAILED') + ':', savefileId);
      ok ? this.onLoadSuccess() : this.onLoadFailure();
    }
  };

  // -------------------------------------------------------------------------
  //  セーブ内容の「安全なクローン」化（非シリアライズ要素の除去・安全版）
  //  元の $game* を直接書き換えず、保存用にクローンを作って不要参照を落とす
  //  ★ nullプロトタイプを Object.prototype にフォールバック（JsonEx対策）
  // -------------------------------------------------------------------------

  // 厳格一致で除外する「危険キー」(ゲーム進行の本体配列等は含めない)
  const HS_BAD_KEYS = new Set([
    '_spine','_spinePlayer','_spineData','_skeleton','_state',
    '_renderer','_sprite','_container','_pixi','_displayObject',
    '_graphics','_runtime','_node'
  ]);

  // PIXI/Spine 実体っぽいものを判定
  function hsIsRuntimeLike(v){
    if (!v || typeof v !== 'object') return false;
    const name = (v.constructor && v.constructor.name) || '';
    if (/^(PIXI|Spine)/i.test(name)) return true;
    if ('worldTransform' in v && ('updateTransform' in v)) return true; // DisplayObject的特徴
    if ('skeleton' in v && 'state' in v) return true;                   // Spine痕跡
    return false;
  }

  // 再帰クローンしながら危険要素を除去（深さ制限あり／null proto→safe proto）
  function hsCloneSanitized(src, depth = 0){
    if (src == null) return src;
    if (depth > 4) return src;                 // 深追いしすぎ防止
    if (typeof src === 'function') return undefined;

    if (Array.isArray(src)){
      return src.map(e => hsCloneSanitized(e, depth + 1));
    }
    if (typeof src === 'object'){
      if (hsIsRuntimeLike(src)) return undefined; // PIXI/Spineなどは落とす

      // ★ 重要：nullプロトタイプを安全な Object.prototype にフォールバック
      const protoOrig = Object.getPrototypeOf(src);
      const protoSafe = protoOrig || Object.prototype;
      const dst = Object.create(protoSafe);

      for (const k of Object.keys(src)){
        if (HS_BAD_KEYS.has(k)) continue;        // 危険キーはスキップ
        const v = hsCloneSanitized(src[k], depth + 1);
        if (v !== undefined) dst[k] = v;
      }
      return dst;
    }
    return src; // プリミティブ
  }

  const _makeSaveContents = DataManager.makeSaveContents;
  DataManager.makeSaveContents = function(){
    const c = _makeSaveContents.call(this);
    try{
      c.system = hsCloneSanitized(c.system);
      c.screen = hsCloneSanitized(c.screen);
      c.map    = hsCloneSanitized(c.map);
      c.player = hsCloneSanitized(c.player);
      c.party  = hsCloneSanitized(c.party);
      c.troop  = hsCloneSanitized(c.troop);
      c.timer  = hsCloneSanitized(c.timer);
      // switches/variables は基本プリミティブ想定なので手を入れない
    }catch(e){
      console.warn('[HS] save sanitizer error:', e);
    }
    return c;
  };

})();

